<?php
namespace App\Http\Controllers;

use App\Models\Donation;
use App\Models\Donor;
use App\Models\Student;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class DonationController extends Controller
{
    // List all donations
    public function index()
    {
        $donations = Donation::with('donor', 'student')->latest()->get();
        return response()->json($donations);
    }

    // Store a new donation
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'donor_id' => 'required|exists:donors,id',
            //'student_id' => 'nullable|exists:students,id',
            'amount' => 'required|numeric|min:1',
              'donation_date' => 'required',
            'description' => 'nullable',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $donation = Donation::create($request->only([
            'donor_id',  'amount', 'description'
        ]));

        return response()->json([
            'message' => 'Donation recorded successfully',
            'data' => $donation
        ]);
    }

    // Show a single donation
    public function show($id)
    {
        $donation = Donation::with('donor', 'student')->findOrFail($id);
        return response()->json($donation);
    }

    // Update donation record
    public function update(Request $request, $id)
    {
        $donation = Donation::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'donor_id' => 'required|exists:donors,id',
            //'student_id' => 'nullable|exists:students,id',
            'amount' => 'required|numeric|min:1',
              'donation_date' => 'required',
            'description' => 'nullable',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $donation->update($request->all());

        return response()->json([
            'message' => 'Donation updated successfully',
            'data' => $donation
        ]);
    }

    // Delete a donation
    public function destroy($id)
    {
        $donation = Donation::findOrFail($id);
        $donation->delete();

        return response()->json(['message' => 'Donation deleted successfully']);
    }

    // Donor's donation history
    public function donationHistory($donor_id)
    {
        $donor = Donor::findOrFail($donor_id);
        $donations = Donation::where('donor_id', $donor_id)
                        ->with('student')
                        ->orderBy('id', 'desc')
                        ->get();

        return response()->json([
            'donor' => $donor,
            'donations' => $donations
        ]);
    }

    // Donor statement: summary report
    public function donorStatement($donor_id)
    {
        $donor = Donor::findOrFail($donor_id);

        $summary = Donation::where('donor_id', $donor_id)
            ->selectRaw('YEAR(donation_date) as year, MONTH(donation_date) as month, SUM(amount) as total_donated')
            ->groupByRaw('YEAR(donation_date), MONTH(donation_date)')
            ->orderByRaw('YEAR(donation_date) DESC, MONTH(donation_date) DESC')
            ->get();

        return response()->json([
            'donor' => $donor,
            'summary' => $summary
        ]);
    }
}

