<?php

namespace App\Http\Controllers\API;
use App\Http\Controllers\Controller;
use App\Models\Student;
use App\Models\StudentFeeRecord;
use App\Models\StudentPersonal;
use App\Models\StudentFamily;
use App\Models\StudentAcademic;
use App\Models\StudentHealth;
use App\Models\StudentAddress;
use Illuminate\Http\Request;
use App\Models\Document;
use Illuminate\Support\Facades\Storage;

class StudentController extends Controller
{
    public function index()
    {
        $students = Student::with('documents', 'donor')->get();
        return response()->json($students);
    }

  public function store(Request $request)
{
    $validated = $request->validate([
        'civil_id'            => 'required|unique:student_personal',
        'roll_number'         => 'required|unique:student_academics',
        'total_fees'          => 'required|numeric',
        'fee_type'            => 'nullable',
        'father_name'         => 'nullable|string',
        'mother_name'         => 'nullable|string',
        'school_address'      => 'nullable|string',
        'class'               => 'nullable',
        'is_special_needs'    => 'nullable|boolean',
        'family_name'         => 'required|string',
        'nationality'         => 'required|string',
        'current_condition'   => 'nullable|string',
        'contact_number'      => 'required|string',
        'home_address'        => 'required|string',
        'home_rent_month'     => 'nullable|numeric',
        'project_type'        => 'nullable|string',
        'child_no'            => 'nullable|integer',
        'number_of_children'  => 'nullable|integer',
        'child_name'          => 'nullable|string',
        'education_level'     => 'required|string',
        'grades'              => 'required|string',
        'school_name'         => 'required|string',
        'donor_id'            => 'nullable|exists:donors,id',
        'status'              => 'required',
        'dob'                 => 'required',
        'academic_year'       => 'nullable|string',
    ]);

    $userId = auth()->id();
    $academicYear = $validated['academic_year'] ?? date('Y') . '-' . (date('Y') + 1);
    unset($validated['academic_year']);

    // Main Student record
    $student = Student::create([
        'created_by' => $userId,
        'updated_by' => $userId
    ]);

    // 1. Student Personal Info
    StudentPersonal::create([
        'student_id'     => $student->id,
        'civil_id'       => $validated['civil_id'],
        'family_name'    => $validated['family_name'],
        'child_name'     => $validated['child_name'] ?? null,
        'dob'            => $validated['dob'] ?? null,
        'nationality'    => $validated['nationality'],
        'contact_number' => $validated['contact_number'],
        'created_by'     => $userId,
        'updated_by'     => $userId
    ]);

    // 2. Student Family Info
    StudentFamily::create([
        'student_id'         => $student->id,
        'father_name'        => $validated['father_name'] ?? null,
        'mother_name'        => $validated['mother_name'] ?? null,
        'child_no'           => $validated['child_no'] ?? null,
        'number_of_children' => $validated['number_of_children'] ?? null,
        'created_by'         => $userId,
        'updated_by'         => $userId
    ]);

    // 3. Student Academics Info
    StudentAcademic::create([
        'student_id'      => $student->id,
        'roll_number'     => $validated['roll_number'],
        'education_level' => $validated['education_level'],
        'grades'          => $validated['grades'],
        'fee_type'        => $validated['fee_type'],
        'total_fees'      => $validated['total_fees'],
        'school_name'     => $validated['school_name'],
        'class'           => $validated['class'] ?? null,
        'created_by'      => $userId,
        'updated_by'      => $userId
    ]);

    // 4. Student Health Info
    StudentHealth::create([
        'student_id'        => $student->id,
        'is_special_needs'  => $validated['is_special_needs'] ?? false,
        'current_condition' => $validated['current_condition'] ?? null,
        'created_by'        => $userId,
        'updated_by'        => $userId
    ]);

    // 5. Student Address
    StudentAddress::create([
        'student_id'      => $student->id,
        'home_address'    => $validated['home_address'],
        'home_rent_month' => $validated['home_rent_month'] ?? null,
        'school_address'  => $validated['school_address'] ?? null,
        'created_by'      => $userId,
        'updated_by'      => $userId
    ]);

    // 6. Student Fee Record
    StudentFeeRecord::create([
        'student_id'     => $student->id,
        'academic_year'  => $academicYear,
        'fee_amount'     => $validated['total_fees'],
        'payment_status' => StudentFeeRecord::STATUS_PENDING,
        'created_by'     => $userId,
        'updated_by'     => $userId
    ]);

    return response()->json([
        'message' => 'Student created successfully',
        'student' => $student->load('personal', 'family', 'academics', 'health', 'address')
    ]);
}


    public function show($id)
    {
        $student = Student::with('documents', 'donor')->findOrFail($id);
        return response()->json($student);
    }

public function update(Request $request, $id)
{
    $validated = $request->validate([
        'civil_id'            => 'required|unique:student_personal,civil_id,' . $id . ',student_id',
        'roll_number'         => 'required|unique:student_academics,roll_number,' . $id . ',student_id',
        'total_fees'          => 'required|numeric',
        'fee_type'            => 'nullable',
        'father_name'         => 'nullable|string',
        'mother_name'         => 'nullable|string',
        'school_address'      => 'nullable|string',
        'class'               => 'nullable',
        'is_special_needs'    => 'nullable|boolean',
        'family_name'         => 'required|string',
        'nationality'         => 'required|string',
        'current_condition'   => 'nullable|string',
        'contact_number'      => 'required|string',
        'home_address'        => 'required|string',
        'home_rent_month'     => 'nullable|numeric',
        'project_type'        => 'nullable|string',
        'child_no'            => 'nullable|integer',
        'number_of_children'  => 'nullable|integer',
        'child_name'          => 'nullable|string',
        'education_level'     => 'required|string',
        'grades'              => 'required|string',
        'school_name'         => 'required|string',
        'donor_id'            => 'nullable|exists:donors,id',
        'status'              => 'required',
        'dob'                 => 'required|date',
        'academic_year'       => 'nullable|string',
    ]);

    $academicYear = $validated['academic_year'] ?? date('Y') . '-' . (date('Y') + 1);
    $userId = auth()->id();

    // Get student
    $student = Student::findOrFail($id);

    // Update Student table
    $student->update([
        'updated_by' => $userId
    ]);

    // Update related tables
    $student->personal->update([
        'civil_id'       => $validated['civil_id'],
        'family_name'    => $validated['family_name'],
        'child_name'     => $validated['child_name'] ?? null,
        'dob'            => $validated['dob'],
        'nationality'    => $validated['nationality'],
        'contact_number' => $validated['contact_number'],
        'updated_by'     => $userId
    ]);

    $student->family->update([
        'father_name'        => $validated['father_name'] ?? null,
        'mother_name'        => $validated['mother_name'] ?? null,
        'child_no'           => $validated['child_no'] ?? null,
        'number_of_children' => $validated['number_of_children'] ?? null,
        'updated_by'         => $userId
    ]);

    $student->academics->update([
        'roll_number'     => $validated['roll_number'],
        'education_level' => $validated['education_level'],
        'grades'          => $validated['grades'],
        'fee_type'        => $validated['fee_type'],
        'total_fees'      => $validated['total_fees'],
        'school_name'     => $validated['school_name'],
        'class'           => $validated['class'] ?? null,
        'updated_by'      => $userId
    ]);

    $student->health->update([
        'is_special_needs'  => $validated['is_special_needs'] ?? false,
        'current_condition' => $validated['current_condition'] ?? null,
        'updated_by'        => $userId
    ]);

    $student->address->update([
        'home_address'    => $validated['home_address'],
        'home_rent_month' => $validated['home_rent_month'] ?? null,
        'school_address'  => $validated['school_address'] ?? null,
        'updated_by'      => $userId
    ]);

    if ($student->feeRecords()->exists()) {
        $student->feeRecords()->latest()->first()->update([
            'academic_year'  => $academicYear,
            'fee_amount'     => $validated['total_fees'],
            'payment_status' => StudentFeeRecord::STATUS_PENDING,
            'updated_by'     => $userId
        ]);
    }

    return response()->json([
        'message' => 'Student updated successfully',
        'student' => $student->load('personal', 'family', 'academics', 'health', 'address', 'feeRecords')
    ]);
}



    public function destroy($id)
    {
        $student = Student::findOrFail($id);
       // $student->documents()->delete();
        $student->delete();

        return response()->json(['message' => 'Student deleted successfully']);
    }

    public function uploadDocument(Request $request, $id)
    {
        $student = Student::findOrFail($id);
        $request->validate([
            'document' => 'required|file|mimes:pdf,jpg,jpeg,png',
        ]);

        $file = $request->file('document');
        $path = $file->store('student_documents', 'public');

        $doc = Document::create([
            'student_id' => $student->id,
            'type'      => $request->type,
            'file_path'  => $path,
            'academic_year' =>$request->academic_year
        ]);

        return response()->json([
            'message'   => 'Document uploaded successfully',
            'document'  => $doc
        ]);
    }

public function autoUpgrade()
{
    $students = Student::with('academic')->get();

    foreach ($students as $student) {
        if (!$student->academic) {
            continue;
        }

        $currentClass = $student->academic->class;

        // If numeric class, just increment
        if (is_numeric($currentClass)) {
            $student->academic->class = (int) $currentClass + 1;
        } else {
            // Handle non-numeric cases (e.g., "LKG" → "UKG")
            $student->academic->class = $currentClass . ' (Promoted)';
        }

        $student->academic->save();
    }

    return response()->json(['message' => 'All students upgraded successfully']);
}
}

